﻿/* Skript zur Demonstration von XML
 * Als Benutzer SCOTT ausführen
 */
 
-- Schrittweise Erzeugung einer XML-Instanz
select xmlelement("Mitarbeiter") resultat
  from emp;

-- Zweiter Schritt: Ergänzung von Attributen
select xmlelement("Mitarbeiter",
         xmlattributes(
           hiredate "einstellDatum",
           empno "id"
         )
       ) resultat
  from emp;
  
-- Dritter Schritt: Ergänzung von Kindelementen
select xmlelement("Mitarbeiter",
         xmlattributes(
           hiredate "einstellDatum",
           empno "id"
         ),
         xmlforest(
           initcap(ename) "Name",
           initcap(job) "Beruf",
           trim(to_char(sal, '999G990D00L')) "Gehalt"
         )
       ) resultat
  from emp;

-- Vierter Schritt: Einkapselung in ein Wurzelelement
select xmlelement("Mitarbeiterliste",
         xmlelement("Mitarbeiter",
           xmlattributes(
             hiredate "einstellDatum",
             empno "id"
           ),
           xmlforest(
             initcap(ename) "Name",
             initcap(job) "Beruf",
             trim(to_char(sal, '999G990D00L')) "Gehalt"
           )
         ) 
       ) resultat
  from emp;
  
-- Fünfter Schritt: Gruppenfunktion
select xmlelement("Mitarbeiterliste",
         xmlagg(
           xmlelement("Mitarbeiter",
             xmlattributes(
               hiredate "einstellDatum",
               empno "id"
             ),
             xmlforest(
               initcap(ename) "Name",
               initcap(job) "Beruf",
               trim(to_char(sal, '999G990D00L')) "Gehalt"
             )
           )
         ) 
       ) resultat
  from emp;
  
-- Letzter Schritt: Gruppierung nach Abteilung

  
-- Fünfter Schritt: Gruppenfunktion
select xmlelement("Mitarbeiterliste",
         xmlattributes(
           deptno "abteilung"
         ),
         xmlagg(
           xmlelement("Mitarbeiter",
             xmlattributes(
               hiredate "einstellDatum",
               empno "id"
             ),
             xmlforest(
               initcap(ename) "Name",
               initcap(job) "Beruf",
               trim(to_char(sal, '999G990D00L')) "Gehalt"
             )
           )
         ) 
       ) resultat
  from emp
 group by deptno;
 
-- Ausgabe mittels XMLSerialize (Einrückung)
select xmlserialize(
         document 
         xmlelement("Mitarbeiterliste",
           xmlattributes(
             deptno "abteilung"
           ),
           xmlagg(
             xmlelement("Mitarbeiter",
               xmlattributes(
                 hiredate "einstellDatum",
                 empno "id"
               ),
               xmlforest(
                 initcap(ename) "Name",
                 initcap(job) "Beruf",
                 trim(to_char(sal, '999G990D00L')) "Gehalt"
               )
             )
           ) 
         ) as clob
         indent size = 3 
       ) resultat
  from emp
 group by deptno;
 
/* Änderungen an XML-Instanzen in SQL durchführen */
-- Erster Schritt: Tabelle, die die vorweg erzeugten XML-Instanzen speichert
create table dept_emp_xml as
select deptno, 
         xmlelement("Mitarbeiterliste",
           xmlattributes(
             deptno "abteilung"
           ),
           xmlagg(
             xmlelement("Mitarbeiter",
               xmlattributes(
                 hiredate "einstellDatum",
                 empno "id"
               ),
               xmlforest(
                 initcap(ename) "Name",
                 initcap(job) "Beruf",
                 trim(to_char(sal, '999G990D00L')) "Gehalt"
               )
             )
           ) 
       ) liste
  from emp
 group by deptno;

select *
  from dept_emp_xml
 where deptno = 10;
  
update dept_emp_xml
   set liste = updatexml(
                 liste,
                 '/Mitarbeiterliste/Mitarbeiter[3]/Beruf/text()',
                 'PRÄSIDENT')
 where deptno = 10;
 
update dept_emp_xml
   set liste = updatexml(liste,
                 '//Mitarbeiter[3]/Beruf',
                 XMLType('<Beruf>PRÄSIDENT</Beruf>'),
                 '//Mitarbeiter[2]/Name',
                 XMLType('<Name>Müller</Name>'))
 where deptno = 10;
 
rollback;
 
-- Löschen eines Elementes
update dept_emp_xml
   set liste = deletexml(liste,
               '//Mitarbeiter[Name="King"]')
 where deptno = 10;
 
select *
  from dept_emp_xml
 where deptno = 10;
 
rollback;

-- Einfügen eines neuen Elementes
-- Ein Geschwisterkind einfügen
update dept_emp_xml
   set liste = 
       insertxmlbefore(liste,
         '//Mitarbeiter[2]',
         xmltype('<Mitarbeiter einstellDatum="12.03.2001">' ||
                 '  <Name>Meier</Name>' ||
                 '  <Beruf>Halbkreisingenieur</Beruf>' ||
                 '  <Gehalt>3.500,00€</Gehalt>' ||
                 '</Mitarbeiter>'))
 where deptno = 10;
 
select liste
  from dept_emp_xml
 where deptno = 10;
 
rollback;
-- Ein Kindelement als letzes anfügen
update dept_emp_xml
   set liste = 
       appendChildXML(liste,
         '/Mitarbeiterliste',
         xmltype('<Mitarbeiter einstellDatum="12.03.2001">' ||
                 '  <Name>Meier</Name>' ||
                 '  <Beruf>Halbkreisingenieur</Beruf>' ||
                 '  <Gehalt>3.500,00€</Gehalt>' ||
                 '</Mitarbeiter>'))
 where deptno = 10;

select liste
  from dept_emp_xml
 where deptno = 10;
 
rollback;
-- Ein Kindelement vor dem ersten Mitarbeiter einfügen
update dept_emp_xml
   set liste = 
       insertChildXMLbefore(liste,
         '/Mitarbeiterliste',
         'Mitarbeiter[1]',
         xmltype('<Mitarbeiter einstellDatum="2001-03-12">' ||
                 '  <Name>Meier</Name>' ||
                 '  <Beruf>Halbkreisingenieur</Beruf>' ||
                 '  <Gehalt>3.500,00€</Gehalt>' ||
                 '</Mitarbeiter>'))
 where deptno = 10;

select liste
  from dept_emp_xml
 where deptno = 10;
 
rollback;

/* Entschachteln der XML-Instanz */
-- Erster Schritt:
select v.*
  from dept_emp_xml x,
       table(
         xmlsequence(
           extract(x.liste, '//Mitarbeiter'))) v
 where deptno = 10;
 
-- Nächster Schritt: Einen Namen extrahieren
select upper(extractValue(v.column_value, '//Name')) ename,
       upper(extractValue(v.column_value, '//Beruf')) job,
       to_date(
         extractValue(
           v.column_value, '/Mitarbeiter/@einstellDatum'),
         'yyyy-mm-dd') hiredate,
       to_number(
         extractValue(
           v.column_value, '//Gehalt'),
         '999G990D00L') sal,
       x.deptno
  from dept_emp_xml x,
       table(
         xmlsequence(
           extract(x.liste, '//Mitarbeiter'))) v;

-- Auswahl einer XML-Instanz mit XMLExists
select liste
  from dept_emp_xml
 where extractValue(
         liste, 
         '//Mitarbeiter[Name="Scott"]/Name') = 'Scott';

/* XQuery */
-- einfaches Beispiel mit einem XPath-Ausdruck
select xmlquery(
       '/Mitarbeiterliste/Mitarbeiter'
       passing liste
       returning content) ergebnis
  from dept_emp_xml;

-- komplexerers Beispiel mit einem FLWOR-Ausdruck
select xmlquery(
       'for $emp in fn:collection("oradb:/SCOTT/EMP")
        where $emp/ROW/SAL >= 3000
        return <Mitarbeiter>
                <Name>{$emp/ROW/ENAME/text()}</Name>
                <Gehalt>{$emp/ROW/SAL/text()}</Gehalt>
               </Mitarbeiter>'
       returning content) ergebnis
  from dual;

-- Erweiteurng auf mehrere Tabellen
select xmlquery(
       'for $emp in fn:collection("oradb:/SCOTT/EMP"),
            $dept in fn:collection("oradb:/SCOTT/DEPT")
        where $emp/ROW/DEPTNO = $dept/ROW/DEPTNO
          and $emp/ROW/SAL >= 3000
        return <Mitarbeiter>
                <Name>{$emp/ROW/ENAME/text()}</Name>
                <Gehalt>{$emp/ROW/SAL/text()}</Gehalt>
                <Abteilung>{$dept/ROW/DNAME/text()}</Abteilung>
               </Mitarbeiter>'
       returning content) ergebnis
  from dual;
  
/* Vorbereitung für die nächste Abfrage */
-- ACL für die EZB an SCOTT granten (als Benutzer SYSTEM ausführen)
begin
  begin
    dbms_network_acl_admin.drop_acl (
      acl          => 'ezb_access.xml');
  exception
    when others then null;
  end;
  dbms_network_acl_admin.create_acl (
    acl          => 'ezb_access.xml', 
    description  => 'Grant to access EZB resources',
    principal    => 'SCOTT',
    is_grant     => true, 
    privilege    => 'connect');
    
  dbms_network_acl_admin.assign_acl (
    acl          => 'ezb_access.xml',
    host         => 'www.ecb.europa.eu', 
    lower_port   => 80,
    upper_port   => 80);
  
  commit;
end;
/

-- Abfrage der XML-Instanz von der EZB
select httpUriType(
         'http://www.ecb.europa.eu/stats/eurofxref/eurofxref-daily.xml').getXml() datei
  from dual;
  
-- Umwandlungsabfrage der Umrechnungskurse
select k.currency, k.rate
  from (select httpUriType(
                 'http://www.ecb.europa.eu/stats/eurofxref/eurofxref-daily.xml').getXml() msg
          from dual) ezb,
       xmltable(
         xmlnamespaces(
           default 'http://www.ecb.int/vocabulary/2002-08-01/eurofxref'),
         '//Cube[@currency]'
         passing ezb.msg
         columns
           currency char(3) path '@currency',
           rate number path 'fn:translate(@rate, ".", ",")'
       ) k;
       
/* Die Verwendung der Funktion XMLExists */
select liste
  from dept_emp_xml
 where xmlexists(
         '/Mitarbeiterliste/Mitarbeiter[Name = "King"]'
         passing liste);

/* Die Verwendung der Funkktion XMLCast */
select xmlcast(
         xmlquery(
           'for $e in //Mitarbeiter
            where $e/Name = "Scott"
            return fn:translate($e/Gehalt, ",.€", ",")'
           passing liste
           returning content) as number) sal
  from dept_emp_xml
 where xmlexists(
         '/Mitarbeiterliste/Mitarbeiter[Name = "Scott"]'
         passing liste);